<?php

namespace Janolaw\Agb\Model;

use \Janolaw\Agb\Model\Term;
use \Magento\Store\Model\ScopeInterface;

/**
 * Download model
 */
class Download extends \Magento\Framework\Model\AbstractModel
{

    const DE = 'de';
    const GB = 'gb';
    const FR = 'fr';

    const TYPE_IMPRINT = 'imprint';
    const TYPE_PRIVACY = 'privacy';
    const TYPE_REVOCATION = 'revocation';
    const TYPE_TAC = 'tac';
    const TYPE_WITHDRAWAL = 'withdrawal';

    const CSS_URL = 'http://www.janolaw.de/agb-service/shops/janolaw.css';
    const CSS_LOCAL_FILE = 'agb.css';

    const CONFIG_PATH_LANGUAGE = 'janolawagb/language/item';

    protected $_types = [
        self::TYPE_IMPRINT,
        self::TYPE_PRIVACY,
        self::TYPE_REVOCATION,
        self::TYPE_TAC,
        self::TYPE_WITHDRAWAL
    ];

    protected $_languages = [
        self::DE => 'German',
        self::GB => 'English',
        self::FR => 'French'
    ];

    protected $_labels = [
        self::TYPE_IMPRINT => [
            self::DE => 'Impressum',
            self::GB => 'Imprint',
            self::FR => 'Mentions légales'
        ],
        self::TYPE_PRIVACY => [
            self::DE => 'Datenschutzerklärung',
            self::GB => 'Data privacy policy',
            self::FR => 'Déclaration quant à la protection des données'
        ],
        self::TYPE_REVOCATION => [
            self::DE => 'Widerrufsbelehrung',
            self::GB => 'Instructions on withdrawal',
            self::FR => 'Informations standardisées sur la rétractation'
        ],
        self::TYPE_TAC => [
            self::DE => 'AGB',
            self::GB => 'General Terms and Conditions',
            self::FR => 'Conditions Générales de Vente'
        ],
        self::TYPE_WITHDRAWAL => [
            self::DE => 'Muster-Widerrufsformular',
            self::GB => 'Model withdrawal form',
            self::FR => 'Modèle de formulaire de rétractation'
        ],
    ];

    protected $_pdf_filename = [
        self::TYPE_IMPRINT => 'legaldetails.pdf',
        self::TYPE_PRIVACY => 'datasecurity.pdf',
        self::TYPE_REVOCATION => 'revocation.pdf',
        self::TYPE_TAC => 'terms.pdf',
        self::TYPE_WITHDRAWAL => 'model-withdrawal-form.pdf'
    ];

    protected $_html_filename = [
        self::TYPE_IMPRINT => 'legaldetails_include.html',
        self::TYPE_PRIVACY => 'datasecurity_include.html',
        self::TYPE_REVOCATION => 'revocation_include.html',
        self::TYPE_TAC => 'terms_include.html',
        self::TYPE_WITHDRAWAL => 'model-withdrawal-form_include.html'
    ];

    protected $_file_helper;
    protected $_current_language;
    protected $_termFactory;
    protected $_demand;
    protected $_scopeConfig;

    /**
     * For logging
     * @var array
     */
    protected $_loggedActions = [];

    /**
     * Constructor
     *
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Janolaw\Agb\Helper\File $fileHelper
     * @param \Janolaw\Agb\Model\TermFactory $termFactory
     * @param \Janolaw\Agb\Model\Demand $demand
     */
    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Janolaw\Agb\Helper\File $fileHelper,
        \Janolaw\Agb\Model\TermFactory $termFactory,
        \Janolaw\Agb\Model\Demand $demand
    ) {
        $this->_file_helper = $fileHelper;
        $this->_scopeConfig = $scopeConfig;
        $this->_termFactory = $termFactory;
        $this->_demand = $demand;

        $this->_current_language = $this->getLanguage();

        parent::__construct($context, $registry);
    }

    /**
     * Main execute
     */
    public function execute()
    {
        $this->executeContent();
        $this->executePdf();
        $this->executeCss();
    }

    /**
     * Get content and update database
     */
    public function executeContent()
    {
        $collection = $this->_termFactory->create()->getCollection();
        $collection->load();

        $termItems = [];
        foreach ($collection as $term) {
            /** @var Term $term */
            if ($term->getId()) {
                $index = $term->getType() . '_' . $term->getLanguage();
                $termItems[$index] = $term;
            }
        }

        $demand = $this->getDemandArray();
        foreach ($demand as $item) {
            $action = '';

            $type = trim($item['type']);
            $language = trim($item['language']);
            $filename = $this->getHtmlFilename($type);

            $index = $type . '_' . $language;
            if (isset($termItems[$index])) {
                $action = 'update';
                $term = $termItems[$index];
            } else {
                $action = 'create';
                $term = $this->_termFactory->create();
            }

            $content = $this->_demand->fetch($filename, $language);

            if ($content) {
                $term->setType($type);
                $term->setLanguage($language);
                $term->setContent($content);
                $term->save();

                //logging
                $this->_loggedActions['content'][$type][$language] = $filename;

                //logging
                if ($action == 'create') {
                    $this->_loggedActions['create'][] = $index;
                }
            } else {
                if (isset($termItems[$index])) {
                    $term->delete();

                    //logging
                    $this->_loggedActions['delete'][] = $index;
                }
            }
        }

        unset($termItems);
    }

    /**
     * Save pdf documents in media/janolawagb_pdf
     * Existed files would be updated
     * @throws \Zend_Pdf_Exception
     */
    public function executePdf()
    {
        $demand = $this->getDemandArray();
        foreach ($demand as $item) {
            $type = trim($item['type']);
            $language = trim($item['language']);
            $filename = $this->getPdfFilename($type);

            $content = $this->_demand->fetch($filename, $language);

            $pdfDir = $this->_file_helper->getLocalPdfBasePath() . '/' . $language;
            $pdfFile = $pdfDir . '/' . $filename;

            if ($content) {
                if (!is_dir($pdfDir)) {
                    mkdir($pdfDir, 0755, true);
                }

                if (file_exists($pdfFile)) {
                    unlink($pdfFile);
                }

                $pdf = \Zend_Pdf::parse($content);
                $pdf->save($pdfFile);

                //logging
                $this->_loggedActions['pdf'][$type][$language] = $filename;
            } else {
                if (file_exists($pdfFile)) {
                    unlink($pdfFile);
                }
            }
        }
    }

    /**
     * Save CSS file in app/code/Janolaw/Agb/view/frontend/web/css/agb.css
     */
    public function executeCss()
    {
        $content = $this->_demand->fetchByUrl(self::CSS_URL);

        if ($content) {
            $fileName = self::CSS_LOCAL_FILE;
            $absModulePath = $this->_file_helper->getAbsModulePath();
            $cssDir = $absModulePath . '/view/frontend/web/css';
            $cssFile = $cssDir . '/' . $fileName;

            if (!is_dir($cssDir)) {
                mkdir($cssDir, 0755, true);
            }

            if (!file_exists($cssFile)) {
                file_put_contents($cssFile, $content);
                chmod($cssFile, 0644);

                //logging
                $this->_loggedActions['css'][] = self::CSS_URL;
            }
        }
    }

    /**
     * Get prepared array for demand
     * @return array
     */
    public function getDemandArray()
    {
        $demand = [];
        foreach ($this->_labels as $type => $labels) {
            foreach ($labels as $language => $label) {
                $demand[] = ['type' => $type, 'language' => $language];
            }
        }

        return $demand;
    }

    /**
     * Get label
     * @param $type
     * @param $language
     * @return mixed
     */
    public function getLabel($type, $language)
    {
        if (isset($this->_labels[$type][$language])) {
            return $this->_labels[$type][$language];
        } else {
            return 'NONE';
        }
    }

    /**
     * Get pdf file name
     * @param $type
     * @return string
     */
    public function getPdfFilename($type)
    {
        if (isset($this->_pdf_filename[$type])) {
            return $this->_pdf_filename[$type];
        } else {
            return 'NONE';
        }
    }

    /**
     * Get html file name
     * @param $type
     * @return string
     */
    public function getHtmlFilename($type)
    {
        if (isset($this->_html_filename[$type])) {
            return $this->_html_filename[$type];
        } else {
            return 'NONE';
        }
    }

    /**
     * Get types
     * @return array
     */
    public function getTypes()
    {
        $return = [];

        foreach ($this->_types as $type) {
            $return[$type] = __($this->getLabel($type, self::GB));
        }
        return $return;
    }

    /**
     * Return existed local pdf file
     * @param $type
     * @param $storeId
     * @return string
     */
    public function getLocalPdfFile($type, $storeId)
    {
        $language = $this->getLanguage($storeId);
        $filename = $this->getPdfFilename($type);

        $pdfDir = $this->_file_helper->getLocalPdfBasePath() . '/' . $language;
        $pdfFile = $pdfDir . '/' . $filename;

        if (!file_exists($pdfFile)) {
            $pdfFile = '';
        }

        return $pdfFile;
    }

    /**
     * @param string $type
     * @param int $storeId
     * @return bool
     */
    public function canAttachPdfByType($type, $storeId)
    {
        $type = trim($type);
        if (!in_array($type, $this->_types)) {
            return false;
        }

        $configPath = 'janolawagb/pdf/' . $type;
        if ($this->_scopeConfig->getValue($configPath, ScopeInterface::SCOPE_STORE, $storeId) == 1) {
            return true;
        }

        return false;
    }

    /**
     * Get languages
     * @return array
     */
    public function getLanguages()
    {
        $return = [];

        foreach ($this->_languages as $language_code => $language_label) {
            $return[$language_code] = __($language_label);
        }
        return $return;
    }

    /**
     * Get default language for backend or language related to store view
     * @param int $storeId
     * @return string
     */
    public function getLanguage($storeId = false)
    {
        if ($storeId === false) {
            $language = $this->_scopeConfig->getValue(self::CONFIG_PATH_LANGUAGE);
        } else {
            $language = $this->_scopeConfig->getValue(self::CONFIG_PATH_LANGUAGE, ScopeInterface::SCOPE_STORE, $storeId);
        }

        return $language;
    }


    /**
     * Get downloaded urls
     */
    public function getLoggedActions()
    {
        return $this->_loggedActions;
    }
}
